import asyncio
import time

from pylog.pylogger import PyLogger


from urpc_enum.corexymoverparameter import CoreXYMoverParameter
from urpc_enum.measurementparameter import MeasurementParameter
from urpc_enum.moverparameter import MoverParameter
from urpc_enum.nodeparameter import NodeParameter
from urpc_enum.serialparameter import SerialParameter
from urpc_enum.temperaturecontrolparameter import TemperatureControlParameter

from fleming.common.firmware_util import *
from fleming.common.node_io import *


#TODO Switch to Virtual Units. For now it was easier to use the firmware_test script instead.


# The delay in seconds for chaning the output in the tests.
DELAY = 5

# Serial Tests #########################################################################################################

async def serial_loopback(serial_name, *txdata) -> bytes:
    serial = get_serial_endpoint(serial_name)
    await serial.SetParameter(SerialParameter.ReadTimeout, 1900, timeout=1)
    await serial.ClearReadBuffer(timeout=1)
    await serial.Write(len(txdata), bytes(txdata), timeout=2)
    rxdata = None
    try:
        response = await serial.Read(len(txdata), timeout=2)
        length = response[0]
        rxdata = response[1:(1 + length)]
    except BaseException as ex:
        PyLogger.logger.error(ex)

    PyLogger.logger.info(f"rxdata: {rxdata}")
    return rxdata


# BCR Tests ############################################################################################################

# from rbartz/firmware_test.py
async def bcr_read(bcr_name, length, timeout=5, sw_trigger=False) -> bytes:
    bcr = get_serial_endpoint(bcr_name)
    node = get_node_endpoint(serial_endpoints[bcr_name]['node'])
    ntrig = serial_endpoints[bcr_name]['ntrig']
    barcode = None
    if 'mux' in serial_endpoints[bcr_name]:
        mux = serial_endpoints[bcr_name]['mux']
        await node.SetDigitalOutput(EEFDigitalOutput.BCREXTMUX0, bool(mux & 1), timeout=1)
        await node.SetDigitalOutput(EEFDigitalOutput.BCREXTMUX1, bool(mux & 2), timeout=1)
        await node.SetDigitalOutput(EEFDigitalOutput.BCREXTMUXEN, 0, timeout=1)

    await bcr.SetParameter(SerialParameter.ReadTimeout, int(timeout * 1000), timeout=1)
    await bcr.ClearReadBuffer(timeout=1)
    if not bool(sw_trigger):
        await node.SetDigitalOutput(ntrig, 0, timeout=1)
    else:
        await bcr.Write(2, b'\x1B\x31', timeout=1)
        response = await bcr.Read(1, timeout=(timeout + 0.1))
        if response[0] < 1 or response[1] != 6:
            PyLogger.logger.error(f"Failed to receive sw_trigger success response {response}")

    try:
        response = await bcr.Read(length, timeout=(timeout + 0.1))
        length = response[0]
        barcode = response[1:(1 + length)]
    except BaseException as ex:
        PyLogger.logger.error(ex)

    if not bool(sw_trigger):
        await node.SetDigitalOutput(ntrig, 1, timeout=1)
        
    PyLogger.logger.info(f"Barcode: {barcode}")
    return barcode


# *********************** PMT *****************************

# Sequencer Tests ######################################################################################################

signals = {
    'none'      : (0),
    'flash'     : (1 << 23),
    'alpha'     : (1 << 22),
    'ingate2'   : (1 << 17),
    'ingate1'   : (1 << 16),
    'hvgate2'   : (1 << 13),
    'hvgate1'   : (1 << 12),
    'hvon3'     : (1 << 10),
    'hvon2'     : (1 <<  9),
    'hvon1'     : (1 <<  8),
    'rstaux'    : (1 <<  6),
    'rstabs'    : (1 <<  5),
    'rstref'    : (1 <<  4),
    'rstpmt2'   : (1 <<  1),
    'rstpmt1'   : (1 <<  0),
}


triggers = {
    'trf'   : (1 << 8),
    'aux'   : (1 << 6),
    'abs'   : (1 << 5),
    'ref'   : (1 << 4),
    'pmt3'  : (1 << 2),
    'pmt2'  : (1 << 1),
    'pmt1'  : (1 << 0),
}

channels = {
    'pmt1'  : (0 << 24),
    'pmt2'  : (1 << 24),
    'pmt3'  : (2 << 24),
    'pmt4'  : (3 << 24),
    'ref'   : (4 << 24),
    'abs'   : (5 << 24),
    'aux'   : (6 << 24),
}



# checks if a level is in the range of a relative acceptance interval around an reference level
# Adds a tolerance of 0.0008 for very small absolut values of less than 0.005.
# returns 1 if level is in the accepance interval around reference
def check_level(level, reference, rel_accept_interval=0.05):
    max_deviation = abs(reference * rel_accept_interval)
    if(reference < 0.005):
        # Add tolerance for very small absolute values
        max_deviation = max_deviation + 0.0008

    deviation = abs(level - reference)
    if(deviation < max_deviation):
      return(RANGE_CHECK_OK)
    else:
      return(RANGE_CHECK_FAIL)

RANGE_CHECK_FAIL = 0; # result of check_level() if level is outside the expected interval
RANGE_CHECK_OK   = 1; # result of check_level() if level is inside the expected interval

RANGE_LOW  = 0; # index to FPGA results for analog result with Range signal is low
RANGE_HIGH = 1; # index to FPGA results for analog result with Range signal is high

# derive the state of the range signal of a PMT or a photo diode by evaluating the analog measurement results.
# the Range digital input is not accessible directly at the FPGA.
# input are two voltage readings, first is the 'analog-low' reading, the second 'analog-high' of the respective sensor.
def get_range_from_voltages(analog_voltages):
    # print(f"info UB get_range_from_voltages({analog_voltages})")
    if(analog_voltages[RANGE_LOW] == 0.0):
        if(analog_voltages[RANGE_HIGH] == 0.0):
            # error if both are zero, a decision can not be made!
            raise Exception(f"get_range_from_voltages({analog_voltages}) -> both voltages are zero! PMT connected to the correct channel?")
        else:
            # ADC-result is delivered as 'analog-high', so the range signal is high
            return RANGE_HIGH
    else:
            # analog-low reading is not zero; analog-high is assumed to be zero then.
            return RANGE_LOW



# Get PMT HV-Monitor - read the I2C ADC on the PMT board
async def get_pmt_hvmon(pmt, verbose = 0):
    meas = get_measurement_endpoint()
    if pmt == 'pmt1' or pmt == 1:
        index = MeasurementParameter.PMT1HighVoltageMonitor
        val = await meas.GetParameter(index, timeout=1)
        # if (verbose != 0): PyLogger.logger.info(f"get_pmt_hvmon() GetParameter = {val}, {type(val)} ")
    elif pmt == 'pmt2' or pmt == 2:
        index = MeasurementParameter.PMT2HighVoltageMonitor
        val = await meas.GetParameter(index, timeout=1)
        result = 'passed'
    else:
        val = -1
        PyLogger.logger.info(f"Param pmt missing = {pmt}")

    spannung_V = val[0]*3.3
    if (verbose != 0): PyLogger.logger.info(f"get_pmt_hvmon() {index} HV-Monitor = {val} -> {spannung_V:.3f} V")

    return spannung_V



# Set PMT HV-Level - set the I2C DAC on the PMT board
# Basis: py_tests/pmt_tests.py
async def set_pmt_hvlev(pmt, level_volt, verbose = 0):
    if (verbose != 0): PyLogger.logger.info(f"setting PMT HV-Level")

    #    await mu.MeasurementFunctions.SetParameter(MeasurementParameter.PMT1DiscriminatorLevel, discr)
    meas = get_measurement_endpoint()
    level = level_volt/3.3
    if (verbose != 0): PyLogger.logger.info(f"set_pmt_hvlev( {pmt}) level_volt  = {level_volt:.3f}, level  = {level}")
    if pmt == 'pmt1' or pmt == 1:
        if (verbose != 0): PyLogger.logger.info(f"  SettingIndex = {MeasurementParameter.PMT1HighVoltageSetting}")
        await meas.SetParameter(MeasurementParameter.PMT1HighVoltageSetting, level, timeout=1)
        #await mu.MeasurementFunctions.SetParameter(MeasurementParameter.PMT1HighVoltageSetting, level)
    if pmt == 'pmt2' or pmt == 2:
        if (verbose != 0): PyLogger.logger.info(f"Bin in  pmt2 = {pmt}")
        await meas.SetParameter(MeasurementParameter.PMT2HighVoltageSetting, level, timeout=1)
    
    

# list defining the test patterns for the Digital output / Analog-Adder test
#                       1500     1000      820 Ohm
DIG_TEST_PATTERNS_REV2 = [['hvgate1','none',   'none'],
                     ['none',   'ingate1','none'],
                     ['none',   'none',   'hvon1'],
                     ['hvgate1','ingate1','none'],
                     ['hvgate1','none',   'hvon1'],
                     ['none',   'ingate1','hvon1'],
                     ['hvgate1','ingate1','hvon1'],
                     ['none',   'none',   'none']]  # last row is expected to be: all signals off

# DIG_TEST_EXPECTED_VOLTAGES = [2.398, 2.218, 1.981, 1.319, 1.082, 0.902, 0.003, 3,297] #  1200     1000      820 Ohm
DIG_TEST_EXPECTED_VOLTAGES_REV2   = [2.339, 2.024, 1.840, 1.356, 1.173, 0.858, 0.191, 3.005] #  1500     1000      820 Ohm based on measurements

# list defining the test patterns for the Digital output / Analog-Adder test
# As of EEF Rev3 the HV-On signal is transmitted to the PMT not by a dedicated wire but via I2C.
# So this signal is not tested here.
#                       1500     1000      820E7 Ohm
DIG_TEST_PATTERNS_REV3 = [['hvgate1','none',   'none'],
                     ['none',   'ingate1','none'],
                     ['hvgate1','ingate1','none'],
                     ['none',   'none',   'none']]  # last row is expected to be: all signals off

# Expected voltages as calculated in 21_08_17_PMT-Tester_EEF-Rev3_AnalogAdder_V1a_Experiment_gemessene_Low_High_Werte.xlsx
DIG_TEST_EXPECTED_VOLTAGES_REV3   = [1.806, 1.477, 0.820, 2.464] #  1500     1000      820E7 Ohm based on measurements

DIG_TEST_PATTERNS_ALL = ['hvgate1','ingate1','hvon1']


# set the test patterns for the Digital output / Analog-Adder test
async def permutate_pmt_dig_pattern(pmt='pmt1', wait_per_step_s = 4):
    for i in range(len(DIG_TEST_PATTERNS_REV2)):
        # print(f"pattern Nr {i}")
        await set_pmt_dig_pattern(pmt, i)
        await asyncio.sleep(wait_per_step_s)



# set the test patterns for the Digital output / Analog-Adder test
async def set_pmt_dig_pattern(pmt='pmt1', pattern_nr = 0):
    PyLogger.logger.info(f"set_pmt_dig_pattern(pattern Nr {pattern_nr}) expected voltage = {DIG_TEST_EXPECTED_VOLTAGES_REV3[pattern_nr]} V")
    set_signal_bitmuster = 0
    for name in DIG_TEST_PATTERNS_REV3[pattern_nr]:
        # print(f"name = {name}")
        set_signal_bitmuster |= signals[name]
    signal_bitmuster_all = 0
    for name in DIG_TEST_PATTERNS_ALL:
        # print(f"name = {name}")
        signal_bitmuster_all |= signals[name]
    if(pmt == 'pmt2'):
        # the pmt2 bit-patterns are located one bit higher than the pmt1 bits:
        set_signal_bitmuster = set_signal_bitmuster << 1
        signal_bitmuster_all = signal_bitmuster_all << 1

    reset_signal_bitmuster = (~set_signal_bitmuster) & signal_bitmuster_all

    # show and apply the patterns to the FPGA digital outputs:
    # print(f" Pattern on: 0x{set_signal_bitmuster:07x} off: 0x{reset_signal_bitmuster:07x} ")
    # print(f" Pattern on: {set_signal_bitmuster:b}b off: {reset_signal_bitmuster:b}b ")
    await pmt_set_signals(set_signal_bitmuster, reset_signal_bitmuster)



# set the digital output 'hvgate1' for PMT1 to high (not used by the pmt_test()???)
#async def pmt_set_sign_HVOnOff():
#    await pmt_set_signals(signals['hvgate1'], signals['ingate1'] | signals['rstpmt1'])
#    return



# set or clear the digital output 'rstpmt1' for PMT1; clear the other outputs (not used by the pmt_test())
async def pmt_set_integr_reset(on = 1):
    if(on == 1):
        await pmt_set_signals(signals['rstpmt1'], signals['ingate1'] | signals['hvgate1'])
    else:
        await pmt_set_signals(0, signals['rstpmt1'] | signals['ingate1'] | signals['hvgate1'])
    return



# from rbartz/firmware_test.py
async def seq_set_signal(name):
    meas = get_measurement_endpoint()
    sequence = [
        0x02000000 | signals[name],     # Set Signal
        0x00000000,
    ]
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0, timeout=1)
    
    return f"seq_set_signal() done"



# from rbartz/firmware_test.py
async def seq_clear_signal(name):
    meas = get_measurement_endpoint()
    sequence = [
        0x03000000 | signals[name],     # Reset Signal
        0x00000000,
    ]
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0, timeout=1)
    
    return f"seq_clear_signal() done"



# set and clear the digital outputs to the PMTs like signals['hvgate1'] etc.
async def pmt_set_signals(set_signal_bitmuster, reset_signal_bitmuster):
    meas = get_measurement_endpoint()
    # print(f" pmt_set_signals({set_signal_bitmuster}, {reset_signal_bitmuster})")

    # !!! ToDo: add check for reset_signal_bitmuster
    if((set_signal_bitmuster & 0xFC000000) != 0):
        PyLogger.logger.error(f"PMT pmt_set_signals error wrong signal_bitmuster = {set_signal_bitmuster}")
    else:
        # compose the FPGA command sequence
        sequence = [
            0x7C000000 | (10000 - 1),                                 # Timer Wait And Restart 100 µs Precounter Window
            0x02000000 | set_signal_bitmuster,                        # Set Signals 
            0x7C000000 | (10000 - 1),                                 # Timer Wait And Restart 100 µs Precounter Window
            0x03000000 | reset_signal_bitmuster,                      # Clear Signals 
            0x00000000,                                               # end
        ]
        for i in range(0, len(sequence), 28):
            subsequence = sequence[i:(i + 28)]
            await meas.WriteSequence(i, len(subsequence), subsequence, timeout=5)

        await meas.StartSequence(0, timeout=1)                        # Start the defined sequence
        # PyLogger.logger.info(f"pmt_set_signals() finished")

    return



async def seq_trf_loopback():
    meas = get_measurement_endpoint()
    sequence = [
        0x01000000 | triggers['trf'],   # Set Trigger Output
        0x74000000 | 1,                 # Wait For Trigger Input
        0x00000000,
    ]
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0, timeout=1)
    await asyncio.sleep(0.1)
    if (await meas.GetStatus())[0] & 0x01:
        return f"seq_trf_loopback() passed"
    else:
        return f"seq_trf_loopback() failed"



# Helper function: defining the FPGA-sequence to read analog measurements (SPI-ADC) of photo diode or PMT peripherals.
# Special sequence without changing digital outputs! Just reads the ADC.
# Intended to get 'integrator' resp. 'analog adder' voltage of the PMT-Simulator.
# Parameter detector may be 'pmt1', 'pmt2' or the photodiode channels 'ref', 'abs' or 'aux'
async def seq_get_analog(detector='pmt1'):
    meas = get_measurement_endpoint()
    reset_delay = 1000                            # 1k * 10 ns = 10 µs
    sequence = [
        0x7C000000 | (reset_delay - 1),           # Timer Start Reset Delay
        # 0x02000000 | signals['rst' + detector], # Set Reset Signal
        0x8C040000,                               # Clear Result Buffer
        0x7C000000 | (10000 - 1),                 # Timer Wait And Restart 100 µs
        # 0x03000000 | signals['rst' + detector], # Clear Reset Signal
    ]
    sequence.extend([
        0x01000000 | triggers[detector],        # Trigger Analog Measurement
        0x80000000 | channels[detector],        # Get Analog High Result
        0x80100001 | channels[detector],        # Get Analog Low Result
        0x00000000,
    ])
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0, timeout=1)
    results = await meas.ReadResults(0, 4, timeout=5)

    analog_results = [i * 5 / 65536 for i in results] # [V] calculate voltage from ADC raw value

    # PyLogger.logger.info(f"alt seq_get_analog({detector}) ; analog_low = {5 / 65536 * results[0]} V ; analog_high = {5 / 65536 * results[1]} V")

    PyLogger.logger.info(f"seq_get_analog({detector}) ; analog_low = {analog_results[0]:.4f} V ; analog_high = {analog_results[1]:.4f} V")

    return analog_results



# Helper function: defining the FPGA-sequence to read analog measurements e.g. for reading photo diode measurements.
# depends on firmware_test signals, triggers and channels
# Parameter detector may be 'pmt1', 'pmt2' or the photodiode channels 'ref', 'abs' or 'aux'
async def seq_get_analog_eef(detector='dummy', window_us=1):
    PyLogger.logger.info(f"pd_test_seq({detector}) entered")
    meas = get_measurement_endpoint()
    PyLogger.logger.info(f"pd_test_seq got meas = {meas}")
    window_coarse, window_fine = divmod(window_us, 65536)
    PyLogger.logger.info(f"pd_test_seq window coarse / fine = {window_coarse} / {window_fine}")
    reset_duration = 100000                        # 1 ms -> time to discharge the integrator capacitor
    sequence = [
        0x7C000000 | (reset_duration - 1),         # Timer Start Reset Delay
        0x02000000 | signals['rst' + detector], # Set Reset Signal
        0x8C040000,                                # Clear Result Buffer
        0x7C000000 | (100 - 1),                    # Timer Wait And Restart 1 µs - prepare timer for the integration loops
        0x03000000 | signals['rst' + detector], # Clear Reset Signal
    ]
    if window_coarse > 0:
        sequence.extend([
            0x07000000 | (window_coarse - 1),   # Loop for window_coarse * 65536 µs
            0x07000000 | (65536 - 1),           # Loop for 65536 * 1 µs
            0x7C000000 | (100 - 1),             # Timer Wait And Restart 1 µs
            0x05000000,                         # Loop End
            0x05000000,                         # Loop End
        ])
    if window_fine > 0:
        sequence.extend([
            0x07000000 | (window_fine - 1),     # Loop for window_fine * 1 µs
            0x7C000000 | (100 - 1),             # Timer Wait And Restart 1 µs
            0x05000000,                         # Loop End
        ])
    sequence.extend([
        0x01000000 | triggers[detector],     # Trigger Analog Measurement (blocking; waits for completion)
        0x80000000 | channels[detector],     # Get Analog High Result
        0x80100001 | channels[detector],     # Get Analog Low Result
        0x00000000,
    ])
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0, timeout=1)      #(blocking; waits for completion)
    results = await meas.ReadResults(0, 4, timeout=5)
    analog_results = [i * 5 / 65536 for i in results] # [V] calculate voltage from ADC raw value

    # PyLogger.logger.info(f"{detector} ; analog_low = {5 / 65536 * results[0]} V ; analog_high = {5 / 65536 * results[1]} V")
    PyLogger.logger.info(f"{detector} ; analog_low = {analog_results[0]} V ; analog_high = {analog_results[1]} V")

    return analog_results




async def seq_get_counter(detector='pmt1', window_us=100):
    meas = get_measurement_endpoint()
    window_corse, window_fine = divmod(window_us, 65536)
    sequence = [
        0x7C000000 | (100 - 1),                 # Timer Wait And Restart 1 µs Precounter Window
        0x88B80000 | channels[detector],        # Reset Precounter and Counter
    ]
    if window_corse > 0:
        sequence.extend([
            0x07000000 | (window_corse - 1),    # Loop for window_corse * 65536 µs
            0x07000000 | (65536 - 1),           # Loop for 65536 * 1 µs
            0x7C000000 | (100 - 1),             # Timer Wait And Restart 1 µs Precounter Window
            0x88D00000 | channels[detector],    # Add Precounter Value to the Counter Value
            0x05000000,                         # Loop End
            0x05000000,                         # Loop End
        ])
    if window_fine > 0:
        sequence.extend([
            0x07000000 | (window_fine - 1),     # Loop for window_fine * 1 µs
            0x7C000000 | (100 - 1),             # Timer Wait And Restart 1 µs Precounter Window
            0x88D00000 | channels[detector],    # Add Precounter Value to the Counter Value
            0x05000000,                         # Loop End
        ])
    sequence.extend([
        0x80900000 | channels[detector],        # Get Pulse Counter Result
        0x00000000,
    ])
    await meas.WriteSequence(0, len(sequence), sequence, timeout=5)
    await meas.StartSequence(0, timeout=1)
    results = await meas.ReadResults(0, 1, timeout=5)
    
    PyLogger.logger.info(f"Count: {detector} = {results[0]:,}")

    return results
